/*
 * $Id: TemplateParameter.java 1349964 2012-06-13 17:18:51Z nlebas $
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.tiles.autotag.model;

import org.apache.tiles.autotag.core.runtime.ModelBody;

/**
 * It represents a parameter in a method in a parsed template class.
 *
 * @version $Rev: 1349964 $ $Date: 2012-06-13 19:18:51 +0200 (Wed, 13 Jun 2012) $
 */
public class TemplateParameter {

    /**
     * The name of the parameter.
     */
    private String name;

    /**
     * The exported name, i.e. the name of the parameter in created code. Usually
     * helpful if this exported name is a reserved word.
     */
    private String exportedName;

    /**
     * The parameter documentation.
     */
    private String documentation;

    /**
     * The type of the parameter.
     */
    private String type;

    /**
     * The default value, as it will be written in Java code.
     */
    private String defaultValue;

    /**
     * Indicates that this parameter is required.
     */
    private boolean required;

    /**
     * Indicates that this parameter is the request.
     */
    private boolean request;

    /**
     * Constructor.
     *
     * @param name The name of the parameter.
     * @param exportedName The exported name, i.e. the name of the parameter in created code. Usually
     * helpful if this exported name is a reserved word.
     * @param type The type of the parameter.
     * @param defaultValue The default value, as it will be written in Java code.
     * @param required Indicates that this parameter is required.
     */
    public TemplateParameter(String name, String exportedName, String type, String defaultValue, boolean required, boolean request) {
        this.name = name;
        this.exportedName = exportedName;
        this.type = type;
        this.defaultValue = defaultValue;
        this.required = required;
        this.request = request;
    }

    /**
     * Returns the documentation for this parameter.
     *
     * @return The documentation.
     */
    public String getDocumentation() {
        return documentation;
    }

    /**
     * Sets the documentation for this parameter.
     *
     * @param documentation The documentation.
     */
    public void setDocumentation(String documentation) {
        this.documentation = documentation;
    }

    /**
     * Returns the name of the parameter.
     *
     * @return The name of the parameter.
     */
    public String getName() {
        return name;
    }

    /**
     * Returns the exported name, i.e. the name of the parameter in created code. Usually
     * helpful if this exported name is a reserved word.
     *
     * @return The exported name.
     */
    public String getExportedName() {
        return exportedName;
    }

    /**
     * Returns the type of the parameter.
     *
     * @return The type.
     */
    public String getType() {
        return type;
    }

    /**
     * Returns the default value, as it will be written in Java code.
     *
     * @return The default value.
     */
    public String getDefaultValue() {
        return defaultValue;
    }

    /**
     * Indicates that this parameter is required.
     *
     * @return <code>true</code> if the parameter is required.
     */
    public boolean isRequired() {
        return required;
    }

    /**
     * Indicates that this parameter implements {@link ModelBody}.
     *
     * @return <code>true</code> if the parameter is a body.
     */
    public boolean isBody() {
        return ModelBody.class.getName().equals(type);
    }

    /**
     * Indicates that this parameter implements {@link Request}.
     *
     * @return <code>true</code> if the parameter is a request.
     */
    public boolean isRequest() {
        return request;
    }

    /**
     * Returns the suffix for getter and setter of the property generated by
     * this parameter.
     *
     * @return The getter and setter suffix.
     */
    public String getGetterSetterSuffix() {
        return exportedName.substring(0, 1).toUpperCase() + exportedName.substring(1);
    }

    @Override
    public String toString() {
        return "TemplateParameter [name=" + name + ", exportedName="
                + exportedName + ", documentation=" + documentation + ", type="
                + type + ", defaultValue=" + defaultValue + ", required="
                + required + ", request=" + request + "]";
    }
}
