/*
 * NASPRO - The NASPRO Architecture for Sound PROcessing
 * Portable runtime library
 *
 * Copyright (C) 2007-2014 Stefano D'Angelo
 *
 * See the COPYING file for license conditions.
 */

/*
   Title: Locales

   Thread-local locales.

   This API allows to control locale settings on a per-thread basis and is
   basically a copy of POSIX's newlocale()/duplocale()/freelocale()/uselocale()
   API.

   These functions only affects the locale settings of the C library and of the
   code based on C library locales. Win32 API users, you have been warned!

   This API is "reasonably" thread-safe, in that it will not change the settings
   of any other thread than the calling thread. However, there is no
   "protection" against other threads calling setlocale() or similar.
 */

#ifndef _NASPRO_CORE_LOCALE_H
#define _NASPRO_CORE_LOCALE_H

#ifndef _NASPRO_CORE_LIB_H
# error Only <NASPRO/core/lib.h> can be included directly.
#endif

NACORE_BEGIN_C_DECLS

/*
   Type: nacore_locale

   Locale object.
 */

/*
   Macro: NACORE_LOCALE_GLOBAL

   Special locale object representing the global locale.

   It is equivalent to LC_GLOBAL_LOCALE on POSIX-compliant platforms.
 */

/*
   Macro: NACORE_LOCALE_COLLATE_MASK

   Mask for the LC_COLLATE locale category.

   It is equivalent to LC_COLLATE_MASK on POSIX-compliant platforms.
 */

/*
   Macro: NACORE_LOCALE_CTYPE_MASK

   Mask for the LC_CTYPE locale category.

   It is equivalent to LC_CTYPE_MASK on POSIX-compliant platforms.
 */

/*
   Macro: NACORE_LOCALE_MONETARY_MASK

   Mask for the LC_MONETARY locale category.

   It is equivalent to LC_MONETARY_MASK on POSIX-compliant platforms.
 */

/*
   Macro: NACORE_LOCALE_NUMERIC_MASK

   Mask for the LC_NUMERIC locale category.

   It is equivalent to LC_NUMERIC_MASK on POSIX-compliant platforms.
 */

/*
   Macro: NACORE_LOCALE_TIME_MASK

   Mask for the LC_TIME locale category.

   It is equivalent to LC_TIME_MASK on POSIX-compliant platforms.
 */

/*
   Macro: NACORE_LOCALE_ALL_MASK

   Mask with bits set for all locale categories.

   Depending on the platform, it may not be equivalent to combining all the
   previous masks together (e.g., on POSIX-compliant platforms it will at least
   also have bits set for LC_MESSAGES).

   It is equivalent to LC_ALL_MASK on POSIX-compliant platforms.
 */

/* Don't mess with this stuff!!! */
#ifdef _WIN32

typedef char *nacore_locale;

_NACORE_DEF extern nacore_locale _nacore_locale_global;
#define NACORE_LOCAL_GLOBAL	_nacore_locale_global

#define NACORE_LOCALE_COLLATE_MASK	0x1
#define NACORE_LOCALE_CTYPE_MASK	(0x1 << 1)
#define NACORE_LOCALE_MONETARY_MASK	(0x1 << 2)
#define NACORE_LOCALE_NUMERIC_MASK	(0x1 << 3)
#define NACORE_LOCALE_TIME_MASK		(0x1 << 4)
#define NACORE_LOCALE_ALL_MASK		0x1f

#else

typedef locale_t nacore_locale;

#define NACORE_LOCALE_GLOBAL		LC_GLOBAL_LOCALE

#define NACORE_LOCALE_COLLATE_MASK	LC_COLLATE_MASK
#define NACORE_LOCALE_CTYPE_MASK	LC_CTYPE_MASK
#define NACORE_LOCALE_MONETARY_MASK	LC_MONETARY_MASK
#define NACORE_LOCALE_NUMERIC_MASK	LC_NUMERIC_MASK
#define NACORE_LOCALE_TIME_MASK		LC_TIME_MASK
#define NACORE_LOCALE_ALL_MASK		LC_ALL_MASK

#endif

/*
   Function: nacore_locale_new()

   Creates a new locale object or modifies an existing one.

   Locale names are platform-dependent and are the same as those accepted by
   setlocale().

   If base is not (nacore_locale)0 and the function succeeds, referring to base
   later will cause undefined behavior.

   The returned locale object should either be released by calling
   <nacore_locale_free()> or be used as a base locale to another
   <nacore_locale_new()> call.

   Do not use NACORE_LOCALE_GLOBAL as base.

   Parameters:

     category_mask	- Mask indicating which categories to be set or
			  modified (i.e., NACORE_LOCALE_ALL_MASK or bitwise OR
			  of other NACORE_LOCALE_*_MASK masks). On
			  POSIX-compliant platforms LC_*_MASK combinations are
			  allowed.
     locale		- Locale name.
     base		- Base locale or (nacore_locale)0 for the current
			  locale.

   Returns:

     New locale object or NULL, in which case errno is set to ENOMEM if there
     was not enough memory or ENOENT if, for any of the categories in
     category_mask, the locale data is not available.
 */
_NACORE_DEF nacore_locale
nacore_locale_new(int category_mask, const char *locale, nacore_locale base);

/*
   Function: nacore_locale_free()

   Releases the resources allocated for a locale object.

   Once this function is called, referring to locale will cause undefined
   behavior.

   Do not use NACORE_LOCALE_GLOBAL as locale.

   Parameters:

     locale	- The locale object.
 */
_NACORE_DEF void
nacore_locale_free(nacore_locale locale);

/*
   Function: nacore_locale_dup()

   Duplicates a locale object.

   The returned locale object should either be released by calling
   <nacore_locale_free()> or be used as a base locale to a <nacore_locale_new()>
   call.

   NACORE_LOCALE_GLOBAL can be used as locale.

   Parameters:

     locale	- The locale object.

   Returns:

     New locale object or NULL if there was not enough memory.
 */
_NACORE_DEF nacore_locale
nacore_locale_dup(nacore_locale locale);

/*
   Function: nacore_locale_use()

   Sets or gets the current locale of the calling thread.

   If locale is not (nacore_locale)0, the specified locale is set for the
   calling thread. If locale is NACORE_GLOBAL_LOCALE, the thread shall use the
   locale determined by setlocale().

   Parameters:

     locale	- The locale object or (nacore_locale)0.

   Returns:

     Reference to the locale object from the previous call for the calling
     thread or NACORE_GLOBAL_LOCALE if there was no such previous call.
 */
_NACORE_DEF nacore_locale
nacore_locale_use(nacore_locale locale);

NACORE_END_C_DECLS

#endif
