package org.biojavax.bio.seq.io;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import junit.framework.TestCase;
import org.biojava.bio.seq.io.SymbolTokenization;
import org.biojavax.Namespace;
import org.biojavax.RichObjectFactory;
import org.biojavax.bio.seq.RichSequence;

/**
 * Tests for EMBLFormat.
 * 
 * @author George Waldon
 */
public class EMBLFormatTest extends TestCase {
    private EMBLFormat emblFormat;

    /**
     * @see junit.framework.TestCase#setUp()
     */
    @Override
    protected void setUp() {
        this.emblFormat = new EMBLFormat();
    }

    /** Test whether the parser reads minimal DNA sequences. The sequence prototype
     * was generated by writing in EMBL format a DNA sequence originally
     * read from a FASTA file (">lcl|test.0| no sequence").
     * Note that EMBL-formatted files require a non-empty accession ; 
     * sequence names are lost and are replaced by accessions. 
     */
    public void testReadEmptyDNASequence() {
        RichSequence sequence = readDNAFile("/empty_embl.embl");
        assertNotNull(sequence);
        assertEquals(sequence.getName(), "test");
        assertEquals(sequence.getAccession(), "test");
        assertEquals(sequence.getVersion(), 0);
        assertEquals(sequence.getDescription(), "no sequence");
        assertEquals(sequence.getInternalSymbolList().length(), 0);
    }

    public void testReadEMBLProteinFile() {
        RichSequence sequence = readProteinFile("/A00060.emblp");
    }

    /**
     * Read a DNA file, return a RichSequence
     * @param filename name of file to read
     * @return a RichSequence instance
     */
    private RichSequence readDNAFile(String filename) {
        InputStream inStream = this.getClass().getResourceAsStream(filename);
        BufferedReader br = new BufferedReader(new InputStreamReader(inStream));
        SymbolTokenization tokenization = RichSequence.IOTools.getDNAParser();
        Namespace namespace = RichObjectFactory.getDefaultNamespace();
        SimpleRichSequenceBuilder builder = new SimpleRichSequenceBuilder();
        RichSequence sequence = null;
        try {
            this.emblFormat.readRichSequence(br, tokenization, builder, namespace);
            sequence = builder.makeRichSequence();
        } catch (Exception e) {
            fail("Unexpected exception: " + e);
        }
        return sequence;
    }

    /**
     * Read a Protein file, return a RichSequence
     * @param filename name of file to read
     * @return a RichSequence instance
     */
    private RichSequence readProteinFile(String filename) {
        InputStream inStream = this.getClass().getResourceAsStream(filename);
        BufferedReader br = new BufferedReader(new InputStreamReader(inStream));
        SymbolTokenization tokenization = RichSequence.IOTools.getProteinParser();
        Namespace namespace = RichObjectFactory.getDefaultNamespace();
        SimpleRichSequenceBuilder builder = new SimpleRichSequenceBuilder();
        RichSequence sequence = null;
        try {
            this.emblFormat.readRichSequence(br, tokenization, builder, namespace);
            sequence = builder.makeRichSequence();
        } catch (Exception e) {
            fail("Unexpected exception: " + e);
        }
        return sequence;
    }
}
